function [betaEst, nuVal] = GLM_DFInv(X, y, targetDF, family, refNu, intercptchoice, wts, dfcalc)
% Compute a ridge estimate to have specified degrees of freedom (DF)
% Arguements:   X -- model matrix, 
%               y -- response vector, 
%               targetDF -- desired degrees of freedom, 
%               family -- GLM family, 
%               refNu -- a reference value for initialization, 
%               intercptchoice -- how the intercept is estimated, 
%               wts -- the weight parameters (not implemented), 
%               dfcalc -- how to compute the df.
% Value: betaEst is the final l2 penalized estimate with the parameter
%       given by nuVal.
% Note that when intercptchoice='simuopt2', this betaEst contains the intercept estimate.
if (nargin == 7) || isempty(dfcalc)
    dfcalc = 'df';
end
if strcmp(intercptchoice, 'simuopt2') 
    X  = [ones(size(X,1), 1),X];
end

if strcmp(dfcalc, 'df') 
    if targetDF > size(X, 2)+ 1e-2 || targetDF > size(X, 1)+ 1e-2 
%         warning('The target DF is too large to be achievable! (usually caused by a tiny ridge parameter)');
%         disp(['targetDF: ', num2str(targetDF), '; size of X: # of rows ', num2str(size(X,1)), ',cols ', num2str(size(X, 2))]);
    end
    targetDF = min(targetDF, min(size(X)));

    if ~strcmp(intercptchoice, 'none') && ~strcmp(intercptchoice, 'simuopt2')
        error('Intercept not implemented yet for DF-inv')
    end
    intercept = 0;

    if wts ~= 0
        error('No weight currently')
    end

    if ~isempty(refNu) & (refNu ~= 0)
        nuUbnd = 1*refNu;
    else
        nuUbnd = 1*norm(X, 2)^2;
    end

    if targetDF < 1e-6
        tmpNu = inf;
        betaEst = zeros(size(X, 2), 1); 
        nuVal = 0;
        return;
    end
    if abs(targetDF - min(size(X))) < 1e-4
        tmpNu = 0;
        if strcmp(intercptchoice, 'none') 
            tmpW = GLM_opt(X, y, tmpNu, family, intercept);
        elseif strcmp(intercptchoice, 'simuopt2') 
            tmpW = GLM_opt(X, y, [0; tmpNu *ones(size(X,2)-1,1)], family, intercept);
        end
        betaEst = tmpW;
        nuVal = tmpNu;    
        return;
    end

    while 1 
        tmpNu = nuUbnd;
        if strcmp(intercptchoice, 'none') 
            tmpW = GLM_opt(X, y, tmpNu, family, intercept);
            tmpDF = hybridTISP_DF(X, y, tmpW, tmpNu, family);
        elseif strcmp(intercptchoice, 'simuopt2') 
            tmpW = GLM_opt(X, y, [0; tmpNu *ones(size(X,2)-1,1)], family, intercept);
            tmpDF = hybridTISP_DF(X, y, tmpW, [0; tmpNu *ones(size(X,2)-1,1)], family);
        end    


        if tmpDF < targetDF    
            break;
        else
            nuUbnd = nuUbnd * 2;
        end
    end

    nuL = 0; nuU = nuUbnd; i = 0;
    while 1
        tmpNu = (nuL + nuU) / 2;
        if strcmp(intercptchoice, 'none') 
            tmpW = GLM_opt(X, y, tmpNu, family, intercept);
            tmpDF = hybridTISP_DF(X, y, tmpW, tmpNu, family);
        elseif strcmp(intercptchoice, 'simuopt2') 
            tmpW = GLM_opt(X, y, [0; tmpNu *ones(size(X,2)-1,1)], family, intercept);
            tmpDF = hybridTISP_DF(X, y, tmpW, [0; tmpNu *ones(size(X,2)-1,1)], family);
        end       

    %     sprintf(['df:', num2str(tmpDF), '; nu:', num2str(tmpNu), '; nuL=', num2str(nuL), ', nuU=', num2str(nuU)])

        if abs(tmpDF-targetDF) < 1e-3 || nuU-nuL<1e-10 % < 1e-2 | nuU-nuL<1e-8 % 
            break;
        else
            if tmpDF < targetDF
                nuU = tmpNu;
            else
                nuL = tmpNu;
            end
        end
        i = i + 1;
    end
elseif strcmp(dfcalc, 'l2')
    
    if ~strcmp(intercptchoice, 'none') && ~strcmp(intercptchoice, 'simuopt2')
        error('Intercept not implemented yet for DF-inv')
    end
    intercept = 0;

    if wts ~= 0
        error('No weight currently')
    end

    if ~isempty(refNu) & (refNu ~= 0)
        nuUbnd = 1*refNu;
    else
        nuUbnd = sum(GLM_opt(X, y, 1e-8, family, intercept).^2); % 1*norm(X, 2)^2;
    end

    if targetDF < 1e-6
        tmpNu = inf;
        betaEst = zeros(size(X, 2), 1); 
        nuVal = 0;
        return;
    end
    while 1 
        tmpNu = nuUbnd;
        if strcmp(intercptchoice, 'none') 
            tmpW = GLM_opt(X, y, tmpNu, family, intercept);
            tmpDF = sum(tmpW.^2); % hybridTISP_DF(X, y, tmpW, tmpNu, family);
        elseif strcmp(intercptchoice, 'simuopt2') 
            tmpW = GLM_opt(X, y, [0; tmpNu *ones(size(X,2)-1,1)], family, intercept);
            tmpDF = sum(tmpW(2:end).^2); % hybridTISP_DF(X, y, tmpW, [0; tmpNu *ones(size(X,2)-1,1)], family);
        end    

        if tmpDF < targetDF    
            break;
        else
            nuUbnd = nuUbnd * 2;
        end
    end

    nuL = 0; nuU = nuUbnd; i = 0;
    while 1
        tmpNu = (nuL + nuU) / 2;
        if strcmp(intercptchoice, 'none') 
            tmpW = GLM_opt(X, y, tmpNu, family, intercept);
            tmpDF = sum(tmpW.^2); %hybridTISP_DF(X, y, tmpW, tmpNu, family);
        elseif strcmp(intercptchoice, 'simuopt2') 
            tmpW = GLM_opt(X, y, [0; tmpNu *ones(size(X,2)-1,1)], family, intercept);
            tmpDF = sum(tmpW(2:end).^2); %hybridTISP_DF(X, y, tmpW, [0; tmpNu *ones(size(X,2)-1,1)], family);
        end       

    %     sprintf(['df:', num2str(tmpDF), '; nu:', num2str(tmpNu), '; nuL=', num2str(nuL), ', nuU=', num2str(nuU)])

        if abs(tmpDF-targetDF) < 1e-3 || nuU-nuL<1e-10 % < 1e-2 | nuU-nuL<1e-8 % 
            break;
        else
            if tmpDF < targetDF
                nuU = tmpNu;
            else
                nuL = tmpNu;
            end
        end
        i = i + 1;
    end
else
    error('not implemented yet for this df calculation')
end
betaEst = tmpW;
nuVal = tmpNu;

% sprintf(['tot # of iters: ', num2str(i), '; targetDF: ', num2str(targetDF), ', approx DF: ', num2str(tmpDF)])
