function [PSNR,PSNR_noise,y_est] = AS_BM3D(nois_img, orig_img, bound_img, obj, sigma)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% Adaptive Shaped BM3D is algorithm for attenuation of additive white Gaussian noise from 
%  grayscale images. This algorithm modifies the results produced in 
%
%  [1] K. Dabov, A. Foi, V. Katkovnik, and K. Egiazarian, "Image denoising 
%   by sparse 3D transform-domain collaborative filtering," 
%   IEEE Transactions on Image Processing, December, 2006, in review,
%   preprint at http://www.cs.tut.fi/~foi/GCF-BM3D.
%
% The function implements the diagram of HT_filter in BM3D and
% bounded Wiener_filter in wiener filter extension. After HT, the original
% design of edge detection on the initial estimation is replace by frames
% been perfectly segmented, in order to test the performance of bounded
% wiener filter extension.
%
%
%
% INPUT ARGUMENTS:
% 1.nois_img (matrix MxN): noisy image 
% 2.orig_img (matrix MxN): original error free image
% 3.bound_img (binary matrix MxN): perfectly segmented binary map of the noisy image, used in wiener filtering. 
% 0 = background; 1 = 1st object; 2 = 2nd object...
% 4.obj(integer): # of objects in bound_img
% 5.sigma: standard deviation of noise
%
%
% OUT ARGUMENTS:
% 1.PSNR: PSNR of y_est
% 2.y_est: denoised image
%
%
% 


%%%% Select transforms ('dct', 'dst', 'hadamard', or anything that is listed by 'help wfilters'):
transform_2D_HT_name     = 'bior1.5'; %% transform used for the HT filt. of size N1 x N1
transform_2D_Wiener_name = 'dct';     %% transform used for the Wiener filt. of size N1_wiener x N1_wiener
transform_3rd_dim_name   = 'haar';    %% transform used in the 3-rd dim, the same for HT and Wiener filt.

%%%% Hard-thresholding (HT) parameters:
N1                  = 8;   %% N1 x N1 is the block size used for the hard-thresholding (HT) filtering
Nstep               = 3;   %% sliding step to process every next reference block
N2                  = 16;  %% maximum number of similar blocks (maximum size of the 3rd dimension of a 3D array)
Ns                  = 39;  %% length of the side of the search neighborhood for full-search block-matching (BM), must be odd
tau_match           = 3000;%% threshold for the block-distance (d-distance)
lambda_thr2D        = 0;   %% threshold parameter for the coarse initial denoising used in the d-distance measure
lambda_thr3D        = 2.7; %% threshold parameter for the hard-thresholding in 3D transform domain
beta                = 2.0; %% parameter of the 2D Kaiser window used in the reconstruction

%%%% Wiener filtering parameters:
N1_wiener           = 8;
Nstep_wiener        = 3;
N2_wiener           = 32;
Ns_wiener           = 39;%odd to make sure the centric symmetry
tau_match_wiener    = 400;
beta_wiener         = 2.0;

%%%% Block-matching parameters:
stepFS              = 1;  %% step that forces to switch to full-search BM, "1" implies always full-search
smallLN             = 'not used in np'; %% if stepFS > 1, then this specifies the size of the small local search neighb.
stepFSW             = 1;
smallLNW            = 'not used in np';
thrToIncStep        = 8;  %% used in the HT filtering to increase the sliding step in uniform regions


decLevel = 0;        %% dec. levels of the dyadic wavelet 2D transform for blocks (0 means full decomposition, higher values decrease the dec. number)
thr_mask = ones(N1); %% N1xN1 mask of threshold scaling coeff. --- by default there is no scaling, however the use of different thresholds for different wavelet decompoistion subbands can be done with this matrix

[Tfor, Tinv]   = getTransfMatrix(N1, transform_2D_HT_name, decLevel);     %% get (normalized) forward and inverse transform matrices
[TforW, TinvW] = getTransfMatrix(N1_wiener, transform_2D_Wiener_name, 0); %% get (normalized) forward and inverse transform matrices

if (strcmp(transform_3rd_dim_name, 'haar') == 1) | (strcmp(transform_3rd_dim_name(end-2:end), '1.1') == 1),
    %%% If Haar is used in the 3-rd dimension, then a fast internal transform is used, thus no need to generate transform
    %%% matrices.
    hadper_trans_single_den         = {};
    inverse_hadper_trans_single_den = {};
else
    %%% Create transform matrices. The transforms are later applied by
    %%% matrix-vector multiplication for the 1D case.
    for hpow = 0:ceil(log2(max(N2,N2_wiener))),
        h = 2^hpow;
        [Tfor3rd, Tinv3rd]   = getTransfMatrix(h, transform_3rd_dim_name, 0);
        hadper_trans_single_den{h}         = single(Tfor3rd);
        inverse_hadper_trans_single_den{h} = single(Tinv3rd');
    end
end

Wwin2D           = kaiser(N1, beta) * kaiser(N1, beta)'; % Kaiser window used in the aggregation of the HT part
Wwin2D_wiener  = kaiser(N1_wiener, beta_wiener) * kaiser(N1_wiener, beta_wiener)'; % Kaiser window used in the aggregation of the Wiener filt. part

PSNR_noise = 10*log10(1/mean((orig_img(:)-nois_img(:)).^2));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% Step 1. Produce the basic estimate by HT filtering
tic;
y_hat = bm3d_thr(nois_img, hadper_trans_single_den, Nstep, N1, N2, lambda_thr2D,...
	lambda_thr3D, tau_match*N1*N1/(255*255), (Ns-1)/2, (sigma/255), thrToIncStep, single(Tfor), single(Tinv)', inverse_hadper_trans_single_den, single(thr_mask), Wwin2D, smallLN, stepFS );

% y_hat = bm3d_BD_HT(nois_img, bound_img, obj, hadper_trans_single_den, Nstep, N1, N2, lambda_thr2D,...
% 	lambda_thr3D, tau_match*N1*N1/(255*255), Ns, (sigma/255), single(Tfor), single(Tinv), Wwin2D);
% y_hat = bm3d_vary_thr(nois_img, orig_img, Nstep, N1, N2, Ns, lambda_thr2D,...
% 	lambda_thr3D, tau_match*N1*N1/(255*255), single(Tfor), single(Tinv), hadper_trans_single_den, Wwin2D)%sigma/255);
 estimate_elapsed_time = toc;
%  edge_hat = edge(y_hat,'canny');
%  figure;
%  imshow(edge_hat);
 
%print initial estimation information
 PSNR_INITIAL_ESTIMATE = 10*log10(1/mean((orig_img(:)-double(y_hat(:))).^2));
 fprintf('BASIC ESTIMATE, PSNR: %.2f dB\n', PSNR_INITIAL_ESTIMATE);
%  figure;
%  imshow(nois_img);
%  figure;
%  imshow(y_hat);
%  figure;
%  e = y_hat-orig_img;
%  imshow(y_hat-orig_img);
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% Step 2. Produce the final estimate by AS_Wiener filtering (using the 
%%%%  hard-thresholding initial estimate)
%%%%
% If no segmentation, use wiener filter   
%    if obj==0     
    tic;
    y_est = bm3d_wiener(nois_img, y_hat, hadper_trans_single_den, Nstep_wiener, N1_wiener, N2_wiener, ...
    'unused arg', tau_match_wiener*N1_wiener*N1_wiener/(255*255), (Ns_wiener-1)/2, (sigma/255), 'unused arg', single(TforW), single(TinvW)', inverse_hadper_trans_single_den, Wwin2D_wiener, smallLNW, stepFSW, single(ones(N1_wiener)) );
%  else
% % If segmented, use bounded wiener filter
%     tic;
%     y_est = BD_wiener(bound_img, obj, y_hat, nois_img, N1_wiener, Nstep_wiener, N2_wiener,...
%     Ns_wiener,(sigma/255), tau_match_wiener*N1_wiener*N1_wiener/(255*255), single(TforW), single(TinvW)', hadper_trans_single_den, Wwin2D_wiener); 
% end

wiener_elapsed_time = toc;
%  y_est = medfilt2(y_est);

PSNR = 10*log10(1/mean((orig_img(:)-y_est(:)).^2)); % y is valid

%     figure, imshow(nois_img); title(sprintf('Noisy, PSNR: %.3f dB (sigma: %d)', ...
%         10*log10(1/mean((orig_img(:)-nois_img(:)).^2)), sigma));
% 
%     figure, imshow(y_est); title(sprintf('Denoised, PSNR: %.3f dB', ...
%         PSNR));

fprintf('FINAL ESTIMATE (total time: %.1f sec), PSNR: %.2f dB\n', ...
        wiener_elapsed_time + estimate_elapsed_time, PSNR);
    